program NELDERMEAD;
{--------------------------------------------------------------------}
{  Alg8'2.pas   Pascal program for implementing Algorithm 8.2        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 8.2 (Nelder-Mead's Minimization Method).                }
{  Section   8.1, Minimization of a Function, Page 414               }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxN = 6;
    Epsilon = 1E-8;
    FunMax = 9;
    MaxV = 200;

  type
    MATRIX = array[0..MaxN, 1..MaxN] of real;
    VECTOR = array[1..MaxN] of real;
    BIGMAT = array[0..MaxV, 1..3] of real;
    BIGVEC = array[0..MaxV] of real;
    LETTER = string[1];
    LETTERS = string[200];
    Status = (Converged, Done, TooBig, Working);
    DoSome = (Go, Stop);

  var
    FunType, InRC, Inum, Lo, Hi, J, K, Count, N, Sub: integer;
    Det, Norm, Rnum: real;
    V: MATRIX;
    Y: VECTOR;
    VA: BIGMAT;
    VY: BIGVEC;
    Ans: CHAR;
    Stat, State: Status;
    DoMo: DoSome;
    Mess: LETTERS;

  function F (P: VECTOR): real;
    var
      X, Y, Z, U, V, W: real;
  begin
    X := P[1];
    Y := P[2];
    Z := P[3];
    U := P[4];
    V := P[5];
    W := P[6];
    case FunType of
      1: 
        F := X * X - 4 * X + Y * Y - Y - X * Y;
      2: 
        F := X * X * X + Y * Y * Y - 3 * X - 3 * Y + 5;
      3: 
        F := X * X + Y * Y + X - 2 * Y - X * Y + 1;
      4: 
        F := X * X * Y + X * Y * Y - 3 * X * Y;
      5: 
        F := X * X * X * X - 8 * X * Y + 2 * Y * Y;
      6: 
        F := (X - Y) / (2 + X * X + Y * Y);
      7: 
        F := X * X * X * X + Y * Y * Y * Y - (X + Y) * (X + Y);
      8: 
        F := (X - Y) * (X - Y) * (X - Y) * (X - Y) + (X + Y - 2) * (X + Y - 2);
      9: 
        F := 100 * (Y - X * X) * (Y - X * X) + (1 - X) * (1 - X);
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer; var N: integer);
    var
      PFUN: string[50];
  begin
    case FunType of
      1: 
        begin
          N := 2;
          PFUN := 'X^2 - 4*X + Y^2 - Y - X*Y';
        end;
      2: 
        begin
          N := 2;
          PFUN := 'X^3 + Y^3 - 3*X - 3*Y + 5';
        end;
      3: 
        begin
          N := 2;
          PFUN := 'X^2 + Y^2 + X - 2*Y - X*Y + 1';
        end;
      4: 
        begin
          N := 2;
          PFUN := 'Y*X^2 + X*Y^2 - 3*X*Y';
        end;
      5: 
        begin
          N := 2;
          PFUN := 'X^4 - 8*X*Y + 2*Y^2';
        end;
      6: 
        begin
          N := 2;
          PFUN := '(X - Y)/(2 + X^2 + Y^2)';
        end;
      7: 
        begin
          N := 2;
          PFUN := 'X^4 + Y^4 - (X+Y)^2';
        end;
      8: 
        begin
          N := 2;
          PFUN := '(X-Y)^4 + (X+Y-2)^2';
        end;
      9: 
        begin
          N := 2;
          PFUN := '100*(Y-X*X)^2 + (1-X)^2';
        end;
    end;
    WRITELN(COPY('F(X,Y,Z,U,V,W', 1, 2 * N + 1), ') = ', PFUN);
  end;

  procedure NELDER ({FUNCTION F(P:VECTOR): real;}
                  var V: MATRIX; var Y: VECTOR; N: integer; Epsilon: real; var Norm: real; var Lo, Hi, Count: integer);
    const
      Min = 10;
      Max = 200;
      Big = 1E11;
    var
      Li, Ho, I, J, K: integer;
      S, YM, YC, YE, YR: real;
      C, E, M, R, Z: VECTOR;

    procedure SIZE;
      var
        J, K: integer;
    begin
      Norm := 0;
      for J := 0 to N do
        begin
          S := 0;
          for K := 1 to N do
            S := S + SQR(V[Lo, K] - V[J, K]);
          if S > Norm then
            Norm := S;
        end;
      Norm := SQRT(Norm);
    end;

    procedure ORDER;
      var
        J: integer;
    begin
      Lo := 0;
      Hi := 0;
      for J := 1 to N do
        begin
          if Y[J] < Y[Lo] then
            Lo := J;
          if Y[J] > Y[Hi] then
            Hi := J;
        end;
      Li := Hi;
      Ho := Lo;
      for J := 0 to N do
        begin
          if (J <> Lo) and (Y[J] < Y[Li]) then
            Li := J;
          if (J <> Hi) and (Y[J] > Y[Ho]) then
            Ho := J;
        end;
    end;

    procedure NEWPOINTS;
      var
        J, K: integer;
    begin
      for K := 1 to N do
        begin
          S := 0;
          for J := 0 to N do
            S := S + V[J, K];
          M[K] := (S - V[Hi, K]) / N;
        end;
      for K := 1 to N do
        R[K] := 2 * M[K] - V[Hi, K];
      YR := F(R);
    end;

    procedure SHRINK;
      var
        J, K: integer;
    begin
      for J := 0 to N do
        begin
          if J <> Lo then
            begin
              for K := 1 to N do
                begin
                  V[J, K] := (V[J, K] + V[Lo, K]) / 2;
                  Z[K] := V[J, K];
                end;
              Y[J] := F(Z);
            end;
        end;
    end;

    procedure REPLACE;
      var
        K: integer;
    begin
      for K := 1 to N do
        V[Hi, K] := R[K];
      Y[Hi] := YR;
    end;

    procedure IMPROVE;
      var
        K: integer;
    begin
      if YR < Y[Ho] then
        begin
          if Y[Li] < YR then
            REPLACE
          else
            begin
              for K := 1 to N do
                E[K] := 2 * R[K] - M[K];
              YE := F(E);
              if YE < Y[Li] then
                begin
                  for K := 1 to N do
                    V[Hi, K] := E[K];
                  Y[Hi] := YE;
                end
              else
                REPLACE;
            end;
        end
      else
        begin
          if YR < Y[Hi] then
            REPLACE;
          for K := 1 to N do
            C[K] := (V[Hi, K] + M[K]) / 2;
          YC := F(C);
          if YC < Y[Hi] then
            begin
              for K := 1 to N do
                V[Hi, K] := C[K];
              Y[Hi] := YC;
            end
          else
            SHRINK;
        end;
    end;

    procedure FUNTOBIG;
      var
        J: integer;
    begin
      for J := 0 to N do
        if ABS(Y[J]) > Big then
          State := TooBig;
    end;

  begin                             {The main part of Procedure Nelder}
    Count := 0;
    ORDER;
    for I := 1 to N do
      VA[Count, I] := V[Lo, I];
    VY[Count] := Y[Lo];
    State := Working;
    while (((Y[Hi] > (Y[Lo] + Epsilon)) and (Count < Max)) or (Count < Min)) and (State = Working) do
      begin
        NEWPOINTS;
        IMPROVE;
        Count := Count + 1;
        ORDER;
        for I := 1 to N do
          VA[Count, I] := V[Lo, I];
        VY[Count] := Y[Lo];
        FUNTOBIG;
      end;
    SIZE;
    if State <> TooBig then
      State := Converged;
  end;                                        {End of Procedure Nelder}


  procedure INPUTVECTORS (var V: MATRIX; var Y: VECTOR; N: integer);

    var
      Count, K, R: integer;
      Z: VECTOR;
  begin
    for R := 0 to N do
      begin
        for K := 1 to N do
          begin
            V[R, K] := 0;
          end;
      end;
    WRITELN;
    WRITELN('     Input the ', (N + 1) : 1, ' vertices of the starting polytope.');
    if (InRC = 1) then
      begin
        for R := 0 to N do
          begin
            WRITELN;
            WRITELN('     ENTER all the coefficients of the vertex  V');
            WRITELN('                                                ', R : 1);
            WRITELN;
            for K := 1 to N do
              Z[K] := 0;
            WRITE('     ');
            case N of
              1: 
                READLN(Z[1]);
              2: 
                READLN(Z[1], Z[2]);
              3: 
                READLN(Z[1], Z[2], Z[3]);
              4: 
                READLN(Z[1], Z[2], Z[3], Z[4]);
              5: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5]);
              6: 
                READLN(Z[1], Z[2], Z[3], Z[4], Z[5], Z[6]);
            end;
            for K := 1 to N do
              begin
                V[R, K] := Z[K];
              end;
            Y[R] := F(Z);
            WRITELN;
          end;
      end
    else
      begin
        for R := 0 to N do
          begin
            WRITELN;
            WRITELN('     ENTER the coefficients of the vertex  V');
            WRITELN('                                            ', R : 1);
            for K := 1 to N do
              begin
                WRITELN;
                WRITE('     V(', R : 1, ',', K : 1, ') = ');
                READLN(V[R, K]);
                Z[K] := V[R, K];
              end;
            Y[R] := F(Z);
          end;
      end;
  end;                                   {End of procedure INPUTMATRIX}

  procedure FACTOR (A: MATRIX; N: integer; var Det: real);
    label
      999;
    type
      SubS = 1..6;
      POINTER0 = array[SubS] of integer;
    var
      C, J, K, P, RowK, RowP, T: integer;
      Row: POINTER0;
  begin
    for J := 1 to N do
      for K := 1 to N do
        A[J, K] := A[J, K] - A[0, K];
    Det := 1;
    for J := 1 to N do                      {Initialize Pointer Vector}
      Row[J] := J;
    for P := 1 to N - 1 do                 {Upper Triangularization Loop}
      begin
        for K := P + 1 to N do
          begin                                            {Find Pivot Row}
            if ABS(A[Row[K], P]) > ABS(A[Row[P], P]) then
              begin
                T := Row[P];
                Row[P] := Row[K];
                Row[K] := T;
                Det := -Det;
              end;
          end;
        Det := Det * A[Row[P], P];
        if Det = 0 then                         {Check Singular Matrix}
          goto 999;
        for K := P + 1 to N do                     {Gaussian Elimination}
          begin
            RowK := Row[K];
            RowP := Row[P];
            A[RowK, P] := A[RowK, P] / A[RowP, P];
            for C := P + 1 to N do
              A[RowK, C] := A[RowK, C] - A[RowK, P] * A[RowP, C];
          end;                                 {End Gaussian Elimination}
999:
      end;                                  {End Upper Triangularization}
    Det := Det * A[Row[N], N];
  end;                                        {End of procedure FACTOR}

  procedure CHANGEVECTORS (var V: MATRIX; var Y: VECTOR; N: integer);
    type
      STATUS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      Count, C, I, K, R: integer;
      Valu: real;
      Z: VECTOR;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) or (Stat = Bad) do
      begin
        CLRSCR;
        WRITELN('         You chose the function:');
        WRITELN;
        WRITE('     ');
        PRINTFUNCTION(FunType, N);
        WRITELN;
        WRITELN('And the starting vertices:');
        WRITELN;
        WRITE('               ');
        for K := 1 to N do
          WRITE('k=', K : 1, '         ');
        WRITELN;
        WRITELN;
        for R := 0 to N do
          begin
            WRITE('V    r=', R : 1);
            for K := 1 to N do
              begin
                WRITE(V[R, K] : 12 : 5);
              end;
            WRITELN;
            WRITELN(' ', R : 1);
          end;
        if Stat = Bad then
          begin
            WRITELN;
            WRITELN('The vertices form a degenerate simplex, you must make a change.');
          end;
        if (Stat <> Bad) then
          begin
            WRITELN;
            WRITE('Do you want to make a change in any vector ?  <Y/N>  ');
            READLN(Resp);
            WRITELN;
          end;
        if (Resp = 'Y') or (Resp = 'y') or (Stat = Bad) then
          begin
            WRITELN;
            WRITELN('     To change a coefficient select');
            case N of
              2: 
                begin
                  WRITELN('        the row    r = 0,1,2');
                  WRITELN('        and column k = 1,2');
                end;
              3: 
                begin
                  WRITELN('        the row    r = 0,1,2,3');
                  WRITELN('        and column k = 1,2,3');
                end;
              else
                begin
                  WRITELN('        the row    r = 0,1,2,...,', N : 2);
                  WRITELN('        and column k = 1,2,...,', N : 2);
                end;
            end;
            Mess := '     ENTER the row r = ';
            WRITE(Mess);
            READLN(R);
            Mess := '     ENTER column  k = ';
            WRITE(Mess);
            READLN(C);
            if (0 <= R) and (R <= N) and (1 <= C) and (C <= N) then
              begin
                WRITELN;
                WRITELN;
                WRITELN('     The current value is V(', R : 1, ',', C : 1, ') =', V[R, C] : 15 : 7);
                WRITELN;
                WRITE('     ENTER the NEW value  V(', R : 1, ',', C : 1, ') = ');
                READLN(V[R, C]);
                WRITELN;
                for R := 0 to N do
                  begin
                    for K := 1 to N do
                      Z[K] := V[R, K];
                    Y[R] := F(Z);
                  end;
              end;
          end
        else
          Stat := Done;
        FACTOR(V, N, Det);
        if ABS(Det) < 0.0000001 then
          Stat := Bad
        else if Stat <> Done then
          Stat := Enter;
      end;
  end;

  procedure GETPOINTS (var V: MATRIX; var Y: VECTOR; var N: integer);
    var
      J, K: integer;
      Num: real;
      Z: VECTOR;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('         You chose the function:');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType, N);
    WRITELN;
    WRITELN('     It is a function of ', N : 1, ' variables. You must supply ', (N + 1) : 1, ' linearly independent');
    WRITELN;
    case N of
      2: 
        begin
          WRITELN('     Starting vertices  V = (v   ,v   )   for  j=0,1,3');
          WRITELN('                         j    j,1  j,2');
        end;
      3: 
        begin
          WRITELN('     Starting vertices  V = (v   ,v   ,v   )   for  j=0,1,3,4');
          WRITELN('                         j    j,1  j,2  j,3');
        end;
      else
        begin
          WRITELN('     Starting vertices  V = (v   ,v   ,...,v   )   for  j=0,1,...,', N);
          WRITELN('                         j    j,1  j,2      j,', N);
        end;
    end;
    WRITELN;
    INPUTVECTORS(V, Y, N);
  end;

  procedure INPUT (var V: MATRIX; var Y: VECTOR; var N: integer);
    var
      I, J, K: integer;
      Num, Valu: real;
      Z: VECTOR;
      Resp: CHAR;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('        Choose how you want to input the coordinates of the vertices.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('    <1> Enter the coefficients on one line separated by spaces, i.e.');
    WRITELN;
    WRITELN('        v     v    ...  v         for j=1,2,...,N');
    WRITELN('         j,1   j,2       j,N ');
    WRITELN;
    WRITELN('    <2> Enter each coefficient on a separate line, i.e.');
    WRITELN;
    WRITELN('        v    ');
    WRITELN('         j,1 ');
    WRITELN('        v    ');
    WRITELN('         j,2 ');
    WRITELN('        .    ');
    WRITELN('        :    ');
    WRITELN('        v        for j=1,2,...,N');
    WRITELN('         j,N ');
    WRITELN;
    Mess := '        SELECT <1 - 2> ?  ';
    InRC := 2;
    WRITE(Mess);
    READLN(InRC);
    if (InRC <> 1) and (InRC <> 2) then
      InRC := 2;
    CLRSCR;
    WRITELN;
    WRITELN('     For convenience, the function f(V) is written using the');
    WRITELN;
    WRITELN('     variables x = v , y = v , z = v , u = v , v = v , w = v .');
    WRITELN('                    1       2       3       4       5       6 ');
    WRITELN;
    WRITELN;
    WRITELN('     That is, f(x,y)');
    WRITELN;
    WRITELN('          or  f(x,y,z)');
    WRITELN;
    WRITELN('          or  f(x,y,z,u)');
    WRITELN('                   .');
    WRITELN('                   :');
    WRITELN('          or  f(x,y,z,u,v,w)');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITE('    Press the <ENTER> key.  ');
    READLN(Resp);
    WRITELN;
    CLRSCR;
    WRITELN;
    WRITELN('    Choose your function to be minimized:');
    I := FunType;
    for I := 1 to FunMax do
      begin
        WRITELN;
        WRITE('    <', I : 2, ' >  ');
        PRINTFUNCTION(I, N);
      end;
    WRITELN;
    WRITE('           SELECT < 1 - ', FunMax : 1, ' > ?  ');
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure OUTPUT (V: MATRIX; var Y: VECTOR; N: integer; Norm: real; Lo, Hi, Count: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('   The Nelder-Mead method was used to find the minimum of the function:');
    WRITELN;
    PRINTFUNCTION(FunType, N);
    WRITELN;
    if State = Converged then
      begin
        WRITELN('It took  ', Count : 3, '  iterations to find an approximation for');
        WRITELN;
        if N = 2 then
          begin
            WRITELN('the coordinates of the local minimum  P = (p ,p )');
            WRITELN('                                            1  2 ');
          end;
        if N = 3 then
          begin
            WRITELN('the coordinates of the local minimum  P = (p ,p ,p )');
            WRITELN('                                            1  2  3 ');
          end;
        if N > 3 then
          begin
            WRITELN('the coordinates of the local minimum  P = (p ,p ,...,p )');
            WRITELN('                                            1  2      ', N);
          end;
      end;
    if State = TooBig then
      begin
        WRITELN('After  ', Count : 3, '  iterations, divergence has been detected, and the');
        WRITELN;
        if N = 2 then
          begin
            WRITELN('coordinates of the last computed point P = (p ,p ) are:');
            WRITELN('                                             1  2      ');
          end;
        if N = 3 then
          begin
            WRITELN('coordinates of the last computed point P = (p ,p ,p ) are:');
            WRITELN('                                             1  2  3      ');
          end;
        if N > 3 then
          begin
            WRITELN('coordinates of the last computed point P = (p ,p ,...,p ) are:');
            WRITELN('                                             1  2      ', N);
          end;
      end;
    for K := 1 to N do
      WRITELN('P(', K : 1, ')  = ', V[Lo, K] : 15 : 7);
    WRITELN;
    WRITELN('  DP  = ', Norm : 15 : 7, '  is the maximum distance to the');
    WRITELN('                            other vertices of the simplex.');
    WRITELN;
    WRITELN('The function value at this point is');
    WRITELN;
    WRITELN('F(P)  = ', Y[Lo] : 15 : 7);
    WRITELN;
    WRITELN('  DF  = ', Y[Hi] - Y[Lo] : 15 : 7, '  is an estimate for the accuracy.');
  end;

  procedure MESSAGE;
  begin
    CLRSCR;
    WRITELN('                          NELDER-MEAD ALGORITHM');
    WRITELN;
    WRITELN('          The Nelder-Mead simplex method or "polytope method"');
    WRITELN;
    WRITELN('     is used to find a local minimum of the function:');
    WRITELN;
    WRITELN;
    WRITELN('              f(v ,v ,...,v )');
    WRITELN('                 1  2      N');
    WRITELN;
    WRITELN('     where V = (v ,v ,...,v )  is a vector of dimension (and N<=6).');
    WRITELN('                 1  2      N');
    WRITELN;
    WRITELN;
    WRITELN('          To start the method, N+1 points must be given which form the');
    WRITELN;
    WRITELN('     vertices of a non-degenerate N-dimensional simplex. They look like:');
    WRITELN;
    WRITELN;
    WRITELN('         V = (v   ,v   ,...,v   )      for j=0,1,...,N');
    WRITELN('          j    j,1  j,2      j,N');
    WRITELN;
    WRITELN;
    WRITE('                          Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('          The function is evaluated at the N+1 vertices and we get');
    WRITELN;
    WRITELN('     values  y  = f(V )  for  j=0,1,...,N.  The "best" N vertices');
    WRITELN('              j      j    ');
    WRITELN('     (where the function is smaller) are retained and one new');
    WRITELN;
    WRITELN('     vertex is formed to take its place. The new vertex is carefully');
    WRITELN;
    WRITELN('     constructed so that the new set of N+1 vertices forms a non-');
    WRITELN;
    WRITELN('     degenerate N-dimensional simplex.');
    WRITELN;
    WRITELN('          Iteration produces a sequence of polytopes which eventually');
    WRITELN;
    WRITELN('     decrease in size and contract toward the point where  f(V)');
    WRITELN;
    WRITELN('     achieves a local minimum.');
    WRITELN;
    WRITELN;
    WRITE('     Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

  procedure PRINTAPPROXS;
    var
      J: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('For each k, the vertex of the simplex where f is minimum are:');
    WRITELN;
    if N = 2 then
      begin
        WRITELN('            (k)  (k)               (k)  (k)                      ');
        WRITELN('  k       (v   ,v   )            (v   ,v   )             f(V ) ');
        WRITELN('            1,1  1,2               2,1  2,2                 k    ');
        WRITELN('-------------------------------------------------------------------------');
      end;
    if N = 3 then
      begin
        WRITELN('            (k)  (k)  (k)          (k)  (k)  (k)          (k)  (k)  (k) ');
        WRITELN('  k       (v   ,v   ,v   )       (v   ,v   ,v   )       (v   ,v   ,v   )');
        WRITELN('            1,1  1,2  1,3          2,1  2,2  2,3          3,1  3,2  3,3 ');
        WRITELN('------------------------------------------------------------------------');
      end;
    for J := 0 to Count do
      begin
        WRITELN;
        if N = 2 then
          WRITELN(' ', J : 2, '     ', VA[J, 1] : 15 : 7, '     ', VA[J, 2] : 15 : 7, '     ', VY[J] : 15 : 7);
        if N = 3 then
          WRITELN(' ', J : 2, '     ', VA[J, 1] : 15 : 7, '     ', VA[J, 2] : 15 : 7, '     ', VA[J, 3] : 15 : 7);
        if (J mod 11 = 8) and (J <> Count) then
          begin
            WRITELN;
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    WRITELN;
  end;

begin                                            {Begin Main Program}
  MESSAGE;
  DoMo := Go;
  while DoMo = Go do
    begin
      INPUT(V, Y, N);
      Stat := Working;
      while Stat = Working do
        begin
          GETPOINTS(V, Y, N);
          CHANGEVECTORS(V, Y, N);
          NELDER(V, Y, N, Epsilon, Norm, Lo, Hi, Count);
          OUTPUT(V, Y, N, Norm, Lo, Hi, Count);
          WRITELN;
          WRITE('Do you want to see  all the approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            PRINTAPPROXS;
          WRITE('Do you want  to try  new starting vertices ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            Stat := Done;
        end;
      WRITELN;
      WRITE('Want  to  minimize  a  different  function ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        DoMo := Stop;
    end;
end.                                            {End of Main Program}

